<?php
/* --------------------------------------------------------------
   Timespan.php 2021-08-01
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Data\Factory;

use DateTimeImmutable;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Timespan as TimespanValueObject;
use Webmozart\Assert\Assert;

class Timespan
{
    private const VALID_TERMS = ["today", "week", "month", "year", "all"];
    
    private const EARLIEST_TIME = "1970-01-01";
    
    
    /**
     * Return time span.
     */
    public function createTimespan(DateTimeImmutable $start, DateTimeImmutable $end): TimespanValueObject
    {
        return TimespanValueObject::create($start, $end);
    }
    
    
    /**
     * Return time span for provided term.
     */
    public function createFromTerm(string $term): TimespanValueObject
    {
        Assert::inArray($term, self::VALID_TERMS);
        
        if ($term === "today") {
            return $this->createForToday();
        }
        
        if ($term === "week") {
            return $this->createForThisWeek();
        }
        
        if ($term === "month") {
            return $this->createForThisMonth();
        }
        
        if ($term === "year") {
            return $this->createForThisYear();
        }
        
        return $this->createForAllTime();
    }
    
    
    /**
     * Return timespan for current day.
     */
    public function createForToday(): TimespanValueObject
    {
        return $this->createTimespan((new DateTimeImmutable())->setTime(0, 0),
                                     (new DateTimeImmutable())->setTime(23, 59, 59));
    }
    
    
    /**
     * Return timespan for current week.
     */
    public function createForThisWeek(): TimespanValueObject
    {
        $monday = new DateTimeImmutable();
        $sunday = new DateTimeImmutable();
        
        // @codeCoverageIgnoreStart
        if ((int)$monday->format('N') !== 1) {
            $monday = $monday->modify('last monday');
        }
        // @codeCoverageIgnoreEnd
    
        $monday = $monday->setTime(0, 0);
        
        // @codeCoverageIgnoreStart
        if ((int)$sunday->format('N') !== 0) {
            $sunday = $sunday->modify('next sunday');
        }
        // @codeCoverageIgnoreEnd
    
        $sunday = $sunday->setTime(23, 59, 59);
        
        return $this->createTimespan($monday, $sunday);
    }
    
    
    /**
     * Return timespan for current month.
     */
    public function createForThisMonth(): TimespanValueObject
    {
        return $this->createTimespan((new DateTimeImmutable())->modify('first day of this month')->setTime(0, 0),
                                     (new DateTimeImmutable())->modify('last day of this month')->setTime(23, 59, 59));
    }
    
    
    /**
     * Return timespan for current year.
     */
    public function createForThisYear(): TimespanValueObject
    {
        return $this->createTimespan((new DateTimeImmutable())->modify('first day of January this year')->setTime(0, 0),
                                     (new DateTimeImmutable())->modify('last day of December this year')
                                         ->setTime(23, 59, 59));
    }
    
    
    /**
     * Return timespan for all time.
     */
    public function createForAllTime(): TimespanValueObject
    {
        return $this->createTimespan(new DateTimeImmutable(self::EARLIEST_TIME),
                                     (new DateTimeImmutable())->setTime(23, 59, 59));
    }
}